function [bnet, names] = mk_orig_bat_dbn()
% MK_BAT_DBN Make the BAT DBN
% [bnet, names] = mk_bat_dbn()
% See
% - Forbes, Huang, Kanazawa and Russell, "The BATmobile: Towards a Bayesian Automated Taxi", IJCAI 95
% - Boyen and Koller, "Tractable Inference for Complex Stochastic Processes", UAI98.

names = {'LeftClr', 'RightClr', 'LatAct', 'Xdot', 'InLane', 'FwdAct', ...
      'Ydot', 'Stopped', 'EngStatus', 'FBStatus', ...
      'LeftClrSens', 'RightClrSens', 'TurnSignalSens', 'XdotSens', 'YdotSens', ...
      'FYdotDiffSens', 'FclrSens', 'BXdotSens', 'BclrSens', 'BYdotDiffSens', ...
      'SensorValid', 'FYdotDiff', 'FcloseSlow', 'Fclr', 'BXdot', 'BcloseFast', 'Bclr', 'BYdotDiff'};
ss = length(names);

intrac = {...
      'LeftClr', 'LeftClrSens';
  'RightClr', 'RightClrSens';
  'LatAct', 'TurnSignalSens'; 'LatAct', 'Xdot';
  'Xdot', 'XdotSens';
  'FwdAct', 'Ydot';
  'Ydot', 'YdotSens'; 'Ydot', 'Stopped';
  'EngStatus', 'Ydot'; 'EngStatus', 'FYdotDiff'; 'EngStatus', 'Fclr'; 'EngStatus', 'BXdot';
  'SensorValid', 'XdotSens';   'SensorValid', 'YdotSens';
  'FYdotDiff', 'FYdotDiffSens'; 'FYdotDiff', 'FcloseSlow';
  'FcloseSlow', 'FBStatus';
  'Fclr', 'FclrSens'; 'Fclr', 'FcloseSlow';
  'BXdot', 'BXdotSens';
  'Bclr', 'BclrSens'; 'Bclr', 'BXdot'; 'Bclr', 'BcloseFast';
  'BcloseFast', 'FBStatus';
  'BYdotDiff', 'BYdotDiffSens'; 'BYdotDiff', 'BcloseFast'};
[intra, names] = mk_adj_mat(intrac, names, 1);


interc = {...
      'LeftClr', 'LeftClr'; 'LeftClr', 'LatAct';
  'RightClr', 'RightClr'; 'RightClr', 'LatAct';
  'LatAct', 'LatAct'; 'LatAct', 'FwdAct';
  'Xdot', 'Xdot'; 'Xdot', 'InLane';
  'InLane', 'InLane'; 'InLane', 'LatAct';
  'FwdAct', 'FwdAct';
  'Ydot', 'Ydot';
  'Stopped', 'Stopped';
  'EngStatus', 'EngStatus';
  'FBStatus', 'FwdAct'; 'FBStatus', 'LatAct'};
inter = mk_adj_mat(interc, names, 0);  

obs = {'LeftClrSens', 'RightClrSens', 'TurnSignalSens', 'XdotSens', 'YdotSens', 'FYdotDiffSens', ...
      'FclrSens', 'BXdotSens', 'BclrSens', 'BYdotDiffSens'};

for i=1:length(obs)
  onodes(i) = stringmatch(obs{i}, names);
end
onodes = sort(onodes);

dnodes = 1:ss; 
ns = zeros(1,ss);

ns(stringmatch('LeftClr', names)) = 2;
ns(stringmatch('RightClr', names)) = 2;
ns(stringmatch('LatAct', names)) = 3;
ns(stringmatch('Xdot', names)) = 7;
ns(stringmatch('InLane', names)) = 2;
ns(stringmatch('FwdAct', names)) = 3;
ns(stringmatch('Ydot', names)) = 11;
ns(stringmatch('Stopped', names)) = 2;
ns(stringmatch('EngStatus', names)) = 2;
ns(stringmatch('FBStatus', names)) = 3;
ns(stringmatch('LeftClrSens', names)) = 2;
ns(stringmatch('RightClrSens', names)) = 2;
ns(stringmatch('TurnSignalSens', names)) = 3;
ns(stringmatch('XdotSens', names)) = 7;
ns(stringmatch('YdotSens', names)) = 11;
ns(stringmatch('FYdotDiffSens', names)) = 8;
ns(stringmatch('FclrSens', names)) = 20;
ns(stringmatch('BXdotSens', names)) = 8;
ns(stringmatch('BclrSens', names)) = 20;
ns(stringmatch('BYdotDiffSens', names)) = 8;
ns(stringmatch('SensorValid', names)) = 2;
ns(stringmatch('FYdotDiff', names)) = 4;
ns(stringmatch('FcloseSlow', names)) = 2;
ns(stringmatch('Fclr', names)) = 3;
ns(stringmatch('BXdot', names)) = 8;
ns(stringmatch('BcloseFast', names)) = 2;
ns(stringmatch('Bclr', names)) = 3;
ns(stringmatch('BYdotDiff', names)) = 4;

%ns = 2*ones(1,ss);


bnet = mk_dbn(intra, inter, ns, 'discrete', dnodes, 'observed', onodes, 'eclass2', (1:ss)+ss);

% make unif params
for i=1:2*ss
  bnet.CPD{i} = tabular_CPD(bnet, i, 'CPT', 'unif');
end

i = stringmatch('LeftClr', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.99 0.01 0.01 0.99]);

i = stringmatch('RightClr', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.99 0.01 0.01 0.99]);

i = stringmatch('LatAct', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.166666666666667 0.975609756097561 0.111111111111111 0.961538461538462 0.00980392156862745 0.0380952380952381 0.00952380952380952 0.037037037037037 0.166666666666667 0.444444444444444 0.0048780487804878 0.0192307692307692 0.0384615384615385 0.888888888888889 0.0344827586206897 0.87719298245614 0.00819672131147541 0.032 0.008 0.03125 0.0384615384615385 0.137931034482759 0.00444444444444444 0.0175438596491228 0.166666666666667 0.975609756097561 0.111111111111111 0.961538461538462 0.166666666666667 0.8 0.04 0.454545454545455 0.166666666666667 0.444444444444444 0.0048780487804878 0.0192307692307692 0.0384615384615385 0.888888888888889 0.0344827586206897 0.87719298245614 0.0384615384615385 0.444444444444444 0.0222222222222222 0.3125 0.0384615384615385 0.137931034482759 0.00444444444444444 0.0175438596491228 0.166666666666667 0.975609756097561 0.111111111111111 0.961538461538462 0.166666666666667 0.8 0.04 0.454545454545455 0.166666666666667 0.444444444444444 0.0048780487804878 0.0192307692307692 0.0384615384615385 0.888888888888889 0.0344827586206897 0.87719298245614 0.0384615384615385 0.444444444444444 0.0222222222222222 0.3125 0.0384615384615385 0.137931034482759 0.00444444444444444 0.0175438596491228 0.666666666666667 0.0195121951219512 0.444444444444444 0.0192307692307692 0.980392156862745 0.952380952380952 0.952380952380952 0.925925925925926 0.666666666666667 0.444444444444444 0.0195121951219512 0.0192307692307692 0.923076923076923 0.106666666666667 0.827586206896552 0.105263157894737 0.983606557377049 0.96 0.96 0.9375 0.923076923076923 0.827586206896552 0.106666666666667 0.105263157894737 0.666666666666667 0.0195121951219512 0.444444444444444 0.0192307692307692 0.666666666666667 0.16 0.16 0.0909090909090909 0.666666666666667 0.444444444444444 0.0195121951219512 0.0192307692307692 0.923076923076923 0.106666666666667 0.827586206896552 0.105263157894737 0.923076923076923 0.533333333333333 0.533333333333333 0.375 0.923076923076923 0.827586206896552 0.106666666666667 0.105263157894737 0.666666666666667 0.0195121951219512 0.444444444444444 0.0192307692307692 0.666666666666667 0.16 0.16 0.0909090909090909 0.666666666666667 0.444444444444444 0.0195121951219512 0.0192307692307692 0.923076923076923 0.106666666666667 0.827586206896552 0.105263157894737 0.923076923076923 0.533333333333333 0.533333333333333 0.375 0.923076923076923 0.827586206896552 0.106666666666667 0.105263157894737 0.166666666666667 0.0048780487804878 0.444444444444444 0.0192307692307692 0.00980392156862745 0.00952380952380952 0.0380952380952381 0.037037037037037 0.166666666666667 0.111111111111111 0.975609756097561 0.961538461538462 0.0384615384615385 0.00444444444444444 0.137931034482759 0.0175438596491228 0.00819672131147541 0.008 0.032 0.03125 0.0384615384615385 0.0344827586206897 0.888888888888889 0.87719298245614 0.166666666666667 0.0048780487804878 0.444444444444444 0.0192307692307692 0.166666666666667 0.04 0.8 0.454545454545455 0.166666666666667 0.111111111111111 0.975609756097561 0.961538461538462 0.0384615384615385 0.00444444444444444 0.137931034482759 0.0175438596491228 0.0384615384615385 0.0222222222222222 0.444444444444444 0.3125 0.0384615384615385 0.0344827586206897 0.888888888888889 0.87719298245614 0.166666666666667 0.0048780487804878 0.444444444444444 0.0192307692307692 0.166666666666667 0.04 0.8 0.454545454545455 0.166666666666667 0.111111111111111 0.975609756097561 0.961538461538462 0.0384615384615385 0.00444444444444444 0.137931034482759 0.0175438596491228 0.0384615384615385 0.0222222222222222 0.444444444444444 0.3125 0.0384615384615385 0.0344827586206897 0.888888888888889 0.87719298245614]);

i = stringmatch('Xdot', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.497512437810945 0.115207373271889 0.0564971751412429 0.0290697674418605 0.075187969924812 0.0300751879699248 0.0298507462686567 0.0980392156862745 0.0392156862745098 0.043956043956044 0.0294117647058824 0.0283687943262411 0.0392156862745098 0.0392156862745098 0.0373134328358209 0.0300751879699248 0.0300751879699248 0.0232558139534884 0.0225988700564972 0.0184331797235023 0.0199004975124378 0.373134328358209 0.460829493087558 0.282485875706215 0.290697674418605 0.37593984962406 0.075187969924812 0.0373134328358209 0.490196078431373 0.0980392156862745 0.043956043956044 0.0294117647058824 0.0283687943262411 0.0392156862745098 0.0392156862745098 0.0746268656716418 0.037593984962406 0.0300751879699248 0.0232558139534884 0.0225988700564972 0.0184331797235023 0.0199004975124378 0.0497512437810945 0.345622119815668 0.564971751412429 0.581395348837209 0.37593984962406 0.37593984962406 0.0746268656716418 0.245098039215686 0.490196078431373 0.21978021978022 0.0735294117647059 0.0354609929078014 0.0490196078431373 0.0392156862745098 0.373134328358209 0.075187969924812 0.037593984962406 0.0232558139534884 0.0225988700564972 0.0184331797235023 0.0199004975124378 0.0199004975124378 0.0230414746543779 0.0282485875706215 0.0290697674418605 0.075187969924812 0.37593984962406 0.373134328358209 0.0490196078431373 0.245098039215686 0.54945054945055 0.735294117647059 0.709219858156028 0.245098039215686 0.0490196078431373 0.373134328358209 0.37593984962406 0.075187969924812 0.0290697674418605 0.0282485875706215 0.0230414746543779 0.0199004975124378 0.0199004975124378 0.0184331797235023 0.0225988700564972 0.0232558139534884 0.037593984962406 0.075187969924812 0.373134328358209 0.0392156862745098 0.0490196078431373 0.0549450549450549 0.0735294117647059 0.141843971631206 0.490196078431373 0.245098039215686 0.0746268656716418 0.37593984962406 0.37593984962406 0.581395348837209 0.564971751412429 0.345622119815668 0.0497512437810945 0.0199004975124378 0.0184331797235023 0.0225988700564972 0.0232558139534884 0.0300751879699248 0.037593984962406 0.0746268656716418 0.0392156862745098 0.0392156862745098 0.043956043956044 0.0294117647058824 0.0283687943262411 0.0980392156862745 0.490196078431373 0.0373134328358209 0.075187969924812 0.37593984962406 0.290697674418605 0.282485875706215 0.460829493087558 0.373134328358209 0.0199004975124378 0.0184331797235023 0.0225988700564972 0.0232558139534884 0.0300751879699248 0.0300751879699248 0.0373134328358209 0.0392156862745098 0.0392156862745098 0.043956043956044 0.0294117647058824 0.0283687943262411 0.0392156862745098 0.0980392156862745 0.0298507462686567 0.0300751879699248 0.075187969924812 0.0290697674418605 0.0564971751412429 0.115207373271889 0.497512437810945]);

i = stringmatch('InLane', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.5 0.3 0.5 0.3 0.5 0.3 0.9 0.01 0.5 0.3 0.5 0.3 0.5 0.3 0.5 0.7 0.5 0.7 0.5 0.7 0.1 0.99 0.5 0.7 0.5 0.7 0.5 0.7]);

i = stringmatch('FwdAct', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.4 0.4 0.4 0.949050949050949 0.949050949050949 0.949050949050949 0.4 0.4 0.4 0.6 0.1 0.1 0.949050949050949 0.949050949050949 0.949050949050949 0.05 0.05 0.05 0.4 0.4 0.4 0.949050949050949 0.949050949050949 0.949050949050949 0.4 0.4 0.4 0.4 0.4 0.4 0.04995004995005 0.04995004995005 0.04995004995005 0.4 0.4 0.4 0.3 0.8 0.3 0.04995004995005 0.04995004995005 0.04995004995005 0.7 0.7 0.7 0.4 0.4 0.4 0.04995004995005 0.04995004995005 0.04995004995005 0.4 0.4 0.4 0.2 0.2 0.2 0.000999000999000999 0.000999000999000999 0.000999000999000999 0.2 0.2 0.2 0.1 0.1 0.6 0.000999000999000999 0.000999000999000999 0.000999000999000999 0.25 0.25 0.25 0.2 0.2 0.2 0.000999000999000999 0.000999000999000999 0.000999000999000999 0.2 0.2 0.2]);

i = stringmatch('Ydot', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.999880014398272 0.72463768115942 0.595238095238095 0.230414746543779 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.595238095238095 0.230414746543779 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.230414746543779 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0144927536231884 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 9.99880014398272e-005 0.144927536231884 0.297619047619048 0.460829493087558 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.297619047619048 0.460829493087558 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.460829493087558 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0144927536231884 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 0.0144927536231884 0.0595238095238095 0.230414746543779 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.0595238095238095 0.230414746543779 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.230414746543779 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0144927536231884 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 9.99880014398272e-006 0.0144927536231884 0.00595238095238095 0.0460829493087558 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0460829493087558 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0460829493087558 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0144927536231884 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0144927536231884 0.00595238095238095 0.00460829493087558 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.00460829493087558 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.00460829493087558 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0144927536231884 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0144927536231884 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00442477876106195 0.00460829493087558 0.00595238095238095 0.0144927536231884 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0144927536231884 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00460829493087558 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00460829493087558 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0442477876106195 0.00460829493087558 0.00595238095238095 0.0144927536231884 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0144927536231884 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0460829493087558 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0460829493087558 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.221238938053097 0.0460829493087558 0.00595238095238095 0.0144927536231884 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0144927536231884 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.230414746543779 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.230414746543779 0.0595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.442477876106195 0.230414746543779 0.0595238095238095 0.0144927536231884 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0144927536231884 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.460829493087558 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.460829493087558 0.297619047619048 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.221238938053097 0.460829493087558 0.297619047619048 0.144927536231884 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0144927536231884 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.230414746543779 0.00595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.230414746543779 0.595238095238095 0.00460829493087558 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.00442477876106195 0.0442477876106195 0.230414746543779 0.595238095238095 0.72463768115942]);

i = stringmatch('Stopped', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.0 0.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0]);

i = stringmatch('EngStatus', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [0.9 1.0e-006 0.1 0.999999]);

i = stringmatch('FBStatus', names)+ss;
bnet.CPD{i} = tabular_CPD(bnet, i, [1.0 0.0 0.0 0.0 0.0 1.0 0.0 1.0 0.0 0.0 1.0 0.0]);

i = stringmatch('SensorValid', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [1.0e-004 0.9999]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [1.0e-004 0.9999]);

i = stringmatch('FYdotDiff', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.908182726364545 0.238095238095238 0.000908182726364545 0.476190476190476 9.08182726364545e-005 0.238095238095238 0.0908182726364545 0.0476190476190476]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.908182726364545 0.238095238095238 0.000908182726364545 0.476190476190476 9.08182726364545e-005 0.238095238095238 0.0908182726364545 0.0476190476190476]);

i = stringmatch('FcloseSlow', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [1.0 1.0 1.0 1.0 1.0 1.0 0.0 1.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0 0.0 1.0 0.0 0.0 0.0 0.0 0.0]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [1.0 1.0 1.0 1.0 1.0 1.0 0.0 1.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0 0.0 1.0 0.0 0.0 0.0 0.0 0.0]);

i = stringmatch('Fclr', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.000499750124937531 0.142857142857143 0.499750124937531 0.285714285714286 0.499750124937531 0.571428571428571]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.000499750124937531 0.142857142857143 0.499750124937531 0.285714285714286 0.499750124937531 0.571428571428571]);

i = stringmatch('BXdot', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.00980392156862745 0.00980392156862745 0.00980392156862745 0.00980392156862745 0.0454545454545455 0.00980392156862745 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.136363636363636 0.0490196078431373 0.196078431372549 0.196078431372549 0.196078431372549 0.196078431372549 0.136363636363636 0.196078431372549 0.392156862745098 0.392156862745098 0.392156862745098 0.392156862745098 0.181818181818182 0.392156862745098 0.196078431372549 0.196078431372549 0.196078431372549 0.196078431372549 0.136363636363636 0.196078431372549 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.136363636363636 0.0490196078431373 0.00980392156862745 0.00980392156862745 0.00980392156862745 0.00980392156862745 0.0454545454545455 0.00980392156862745 0.0980392156862745 0.0980392156862745 0.0980392156862745 0.0980392156862745 0.181818181818182 0.0980392156862745]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.00980392156862745 0.00980392156862745 0.00980392156862745 0.00980392156862745 0.0454545454545455 0.00980392156862745 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.136363636363636 0.0490196078431373 0.196078431372549 0.196078431372549 0.196078431372549 0.196078431372549 0.136363636363636 0.196078431372549 0.392156862745098 0.392156862745098 0.392156862745098 0.392156862745098 0.181818181818182 0.392156862745098 0.196078431372549 0.196078431372549 0.196078431372549 0.196078431372549 0.136363636363636 0.196078431372549 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.0490196078431373 0.136363636363636 0.0490196078431373 0.00980392156862745 0.00980392156862745 0.00980392156862745 0.00980392156862745 0.0454545454545455 0.00980392156862745 0.0980392156862745 0.0980392156862745 0.0980392156862745 0.0980392156862745 0.181818181818182 0.0980392156862745]);

i = stringmatch('BcloseFast', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0]);

i = stringmatch('Bclr', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.142857142857143 0.285714285714286 0.571428571428571]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.142857142857143 0.285714285714286 0.571428571428571]);

i = stringmatch('BYdotDiff', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.238095238095238 0.476190476190476 0.238095238095238 0.0476190476190476]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.238095238095238 0.476190476190476 0.238095238095238 0.0476190476190476]);

i = stringmatch('LeftClrSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.909090909090909 0.0909090909090909 0.0909090909090909 0.909090909090909]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.909090909090909 0.0909090909090909 0.0909090909090909 0.909090909090909]);

i = stringmatch('RightClrSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.909090909090909 0.0909090909090909 0.0909090909090909 0.909090909090909]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.909090909090909 0.0909090909090909 0.0909090909090909 0.909090909090909]);

i = stringmatch('TurnSignalSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.75 0.000998003992015968 0.01 0.24 0.998003992015968 0.24 0.01 0.000998003992015968 0.75]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.75 0.000998003992015968 0.01 0.24 0.998003992015968 0.24 0.01 0.000998003992015968 0.75]);

i = stringmatch('XdotSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.142857142857143 0.897666068222621 0.142857142857143 0.0824402308326463 0.142857142857143 0.00818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.000824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.0897666068222621 0.142857142857143 0.824402308326463 0.142857142857143 0.0818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.000824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.00897666068222621 0.142857142857143 0.0824402308326463 0.142857142857143 0.818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.000824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.00824402308326463 0.142857142857143 0.0818330605564648 0.142857142857143 0.818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.00824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.000824402308326463 0.142857142857143 0.00818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.818330605564648 0.142857142857143 0.0824402308326463 0.142857142857143 0.00897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.000824402308326463 0.142857142857143 0.000818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.824402308326463 0.142857142857143 0.0897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.000824402308326463 0.142857142857143 0.000818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.0824402308326463 0.142857142857143 0.897666068222621]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.142857142857143 0.897666068222621 0.142857142857143 0.0824402308326463 0.142857142857143 0.00818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.000824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.0897666068222621 0.142857142857143 0.824402308326463 0.142857142857143 0.0818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.000824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.00897666068222621 0.142857142857143 0.0824402308326463 0.142857142857143 0.818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.000824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.00824402308326463 0.142857142857143 0.0818330605564648 0.142857142857143 0.818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.00824402308326463 0.142857142857143 0.000897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.000824402308326463 0.142857142857143 0.00818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.818330605564648 0.142857142857143 0.0824402308326463 0.142857142857143 0.00897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.000824402308326463 0.142857142857143 0.000818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.0818330605564648 0.142857142857143 0.824402308326463 0.142857142857143 0.0897666068222621 0.142857142857143 0.000897666068222621 0.142857142857143 0.000824402308326463 0.142857142857143 0.000818330605564648 0.142857142857143 0.000818330605564648 0.142857142857143 0.00818330605564648 0.142857142857143 0.0824402308326463 0.142857142857143 0.897666068222621]);

i = stringmatch('YdotSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.0909090909090909 0.894454382826476 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.0894454382826476 0.0909090909090909 0.821692686935086 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.00894454382826476 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.00821692686935086 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.00894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.821692686935086 0.0909090909090909 0.0894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.894454382826476]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.0909090909090909 0.894454382826476 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.0894454382826476 0.0909090909090909 0.821692686935086 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.00894454382826476 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.00821692686935086 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.00821692686935086 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.815660685154976 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.00894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0815660685154976 0.0909090909090909 0.821692686935086 0.0909090909090909 0.0894454382826476 0.0909090909090909 0.000894454382826476 0.0909090909090909 0.000821692686935086 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.000815660685154976 0.0909090909090909 0.00815660685154975 0.0909090909090909 0.0821692686935086 0.0909090909090909 0.894454382826476]);

i = stringmatch('FYdotDiffSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.53191206429753 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.265956032148765 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.132978016074383 0.0806445109313634 0.0053191206429753 9.9930048965724e-005 0.053191206429753 0.806445109313635 0.053191206429753 9.9930048965724e-005 0.0053191206429753 0.0806445109313634 0.132978016074383 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.265956032148765 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.53191206429753 9.9930048965724e-005 5.3191206429753e-006 8.06445109313635e-006 5.3191206429753e-006 0.99930048965724]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.53191206429753 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.265956032148765 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.132978016074383 0.0806445109313634 0.0053191206429753 9.9930048965724e-005 0.053191206429753 0.806445109313635 0.053191206429753 9.9930048965724e-005 0.0053191206429753 0.0806445109313634 0.132978016074383 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.265956032148765 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.53191206429753 9.9930048965724e-005 5.3191206429753e-006 8.06445109313635e-006 5.3191206429753e-006 0.99930048965724]);

i = stringmatch('FclrSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.472366556447804 0.0044762757385855 6.20655412115194e-005 0.472366556447804 0.044762757385855 6.20655412115194e-005 0.0472366556447804 0.44762757385855 0.000620655412115194 0.000472366556447804 0.44762757385855 0.00620655412115194 0.000472366556447804 0.044762757385855 0.0620655412115194 0.000472366556447804 0.0044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.472366556447804 0.0044762757385855 6.20655412115194e-005 0.472366556447804 0.044762757385855 6.20655412115194e-005 0.0472366556447804 0.44762757385855 0.000620655412115194 0.000472366556447804 0.44762757385855 0.00620655412115194 0.000472366556447804 0.044762757385855 0.0620655412115194 0.000472366556447804 0.0044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194]);

i = stringmatch('BXdotSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.689655172413793 0.172413793103448 0.0546448087431694 0.00546448087431694 0.00546448087431694 0.00546448087431694 0.00574712643678161 0.00689655172413793 0.206896551724138 0.574712643678161 0.163934426229508 0.0546448087431694 0.00546448087431694 0.00546448087431694 0.00574712643678161 0.00689655172413793 0.0689655172413793 0.172413793103448 0.546448087431694 0.163934426229508 0.0546448087431694 0.00546448087431694 0.00574712643678161 0.00689655172413793 0.00689655172413793 0.0574712643678161 0.163934426229508 0.546448087431694 0.163934426229508 0.0546448087431694 0.00574712643678161 0.00689655172413793 0.00689655172413793 0.00574712643678161 0.0546448087431694 0.163934426229508 0.546448087431694 0.163934426229508 0.0574712643678161 0.00689655172413793 0.00689655172413793 0.00574712643678161 0.00546448087431694 0.0546448087431694 0.163934426229508 0.546448087431694 0.172413793103448 0.0689655172413793 0.00689655172413793 0.00574712643678161 0.00546448087431694 0.00546448087431694 0.0546448087431694 0.163934426229508 0.574712643678161 0.206896551724138 0.00689655172413793 0.00574712643678161 0.00546448087431694 0.00546448087431694 0.00546448087431694 0.0546448087431694 0.172413793103448 0.689655172413793]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.689655172413793 0.172413793103448 0.0546448087431694 0.00546448087431694 0.00546448087431694 0.00546448087431694 0.00574712643678161 0.00689655172413793 0.206896551724138 0.574712643678161 0.163934426229508 0.0546448087431694 0.00546448087431694 0.00546448087431694 0.00574712643678161 0.00689655172413793 0.0689655172413793 0.172413793103448 0.546448087431694 0.163934426229508 0.0546448087431694 0.00546448087431694 0.00574712643678161 0.00689655172413793 0.00689655172413793 0.0574712643678161 0.163934426229508 0.546448087431694 0.163934426229508 0.0546448087431694 0.00574712643678161 0.00689655172413793 0.00689655172413793 0.00574712643678161 0.0546448087431694 0.163934426229508 0.546448087431694 0.163934426229508 0.0574712643678161 0.00689655172413793 0.00689655172413793 0.00574712643678161 0.00546448087431694 0.0546448087431694 0.163934426229508 0.546448087431694 0.172413793103448 0.0689655172413793 0.00689655172413793 0.00574712643678161 0.00546448087431694 0.00546448087431694 0.0546448087431694 0.163934426229508 0.574712643678161 0.206896551724138 0.00689655172413793 0.00574712643678161 0.00546448087431694 0.00546448087431694 0.00546448087431694 0.0546448087431694 0.172413793103448 0.689655172413793]);

i = stringmatch('BclrSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.472366556447804 0.0044762757385855 6.20655412115194e-005 0.472366556447804 0.044762757385855 6.20655412115194e-005 0.0472366556447804 0.44762757385855 0.000620655412115194 0.000472366556447804 0.44762757385855 0.00620655412115194 0.000472366556447804 0.044762757385855 0.0620655412115194 0.000472366556447804 0.0044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.472366556447804 0.0044762757385855 6.20655412115194e-005 0.472366556447804 0.044762757385855 6.20655412115194e-005 0.0472366556447804 0.44762757385855 0.000620655412115194 0.000472366556447804 0.44762757385855 0.00620655412115194 0.000472366556447804 0.044762757385855 0.0620655412115194 0.000472366556447804 0.0044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194 0.000472366556447804 0.00044762757385855 0.0620655412115194]);

i = stringmatch('BYdotDiffSens', names);
%bnet.CPD{i} = tabular_CPD(bnet, i, [0.53191206429753 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.265956032148765 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.132978016074383 0.0806445109313634 0.0053191206429753 9.9930048965724e-005 0.053191206429753 0.806445109313635 0.053191206429753 9.9930048965724e-005 0.0053191206429753 0.0806445109313634 0.132978016074383 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.265956032148765 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.53191206429753 9.9930048965724e-005 5.3191206429753e-006 8.06445109313635e-006 5.3191206429753e-006 0.99930048965724]);
bnet.CPD{i+ss} = tabular_CPD(bnet, i+ss, [0.53191206429753 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.265956032148765 0.00806445109313635 0.0053191206429753 9.9930048965724e-005 0.132978016074383 0.0806445109313634 0.0053191206429753 9.9930048965724e-005 0.053191206429753 0.806445109313635 0.053191206429753 9.9930048965724e-005 0.0053191206429753 0.0806445109313634 0.132978016074383 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.265956032148765 9.9930048965724e-005 0.0053191206429753 0.00806445109313635 0.53191206429753 9.9930048965724e-005 5.3191206429753e-006 8.06445109313635e-006 5.3191206429753e-006 0.99930048965724]);
%BIF2BNT added a bunch of zeros at the end of this cpd. Hopefully the only occurence of this bug!  0 0 0 0 0 0 0 0]);